/* Code for sending IR commands to a Syma Micro Helicopter using serial port data
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <stdint.h>
#include <util/delay.h>
#include "usb_serial.h"

//sync pin helps wth triggering oscilloscope
#define SYNC_CONFIG	(DDRD |= (1<<5))
#define SYNC_ON		(PORTD |= (1<<5))
#define SYNC_OFF	(PORTD &= ~(1<<5))

//onboard status LED
#define LED_CONFIG	(DDRD |= (1<<6))
#define LED_ON		(PORTD |= (1<<6))
#define LED_OFF		(PORTD &= ~(1<<6))
#define CPU_PRESCALE(n) (CLKPR = 0x80, CLKPR = (n))

//IR emitter
#define IR_CONFIG	(DDRD |= (1<<0))
#define IR_ON		(PORTD |= (1<<0))
#define IR_OFF		(PORTD &= ~(1<<0))

void send_str(const char *s);
uint8_t recv_str(char *buf, uint8_t size);
void parse_and_execute_command(const char *buf, uint8_t num);
int pulse_length = 0;

void pulseIR(int duration_us){
	//38Khz	
	while(duration_us > 0) {
		IR_ON;
		_delay_us(13);	
		IR_OFF;
		IR_OFF;
		IR_OFF;
		IR_OFF;
		IR_OFF;
		_delay_us(12);	
		duration_us -= 26;
	}
}

void SendIRValue(int v) {
	for(int i = 7; i >= 0; i--) {
		if(((v>>i)&0x01)==1) {
		  pulseIR(320);
		  _delay_us(680);
		  pulse_length += 1000;		
		} else {
		  pulseIR(320);
		  _delay_us(280);
		  pulse_length += 600;
		}
	}
}

// Very simple character echo test
int main(void)
{
	CPU_PRESCALE(0);
	LED_CONFIG;
	IR_CONFIG;
	SYNC_CONFIG;
	usb_init();

	int yaw = 63;
	int pitch = 63;
	int throttle = 0;
	int yaw_correction = 63;
	int count = 0;
	LED_ON;
	int timeout_cycles = 5;
	while (1) {

		if(count < timeout_cycles) {
			if(throttle == 0) {
				yaw = 63;
				pitch = 63;
			}
			//only send messages if we have recieved a command recently
			SYNC_ON;
			cli();//disable interrupts
			pulseIR(2000);
			SYNC_OFF;
			_delay_us(2000);
			pulse_length = 4000;
			SendIRValue(yaw);
			SendIRValue(pitch);
			SendIRValue(throttle);
			SendIRValue(yaw_correction);
	 		pulseIR(300);//terminating bit
			sei(); //re-enable interrupts

			//stock controller sends out commands every 120ms	
			//you can try reducing this until it starts to stutter
			_delay_us( 120000 - pulse_length);
			count++;
		} else {
			//we haven't recieved an update in a few cycles, stop sending
			throttle = 0;
		}

		if(usb_serial_available()) {
			int n = usb_serial_getchar();
			if (n == 255) { //header byte
				//the following should only range between 0-127
				yaw = usb_serial_getchar();
				pitch = usb_serial_getchar();
				throttle = usb_serial_getchar();
				yaw_correction = usb_serial_getchar();
				usb_serial_putchar_nowait('k');//ack
				usb_serial_flush_input();//get rid of anything left
				count = 0;
			}
		}
	}
}
