// Team "Funktion im Kopf der Mensch"

#include <stdio.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>

#include "um.h"

#define MAX_ARRAYS		128000000

typedef struct _array_t
{
    platter_t len;
    platter_t platters[0];
} array_t;

static int debug_mon;

static array_t *arrays[MAX_ARRAYS];
static platter_t regs[NUM_REGS];
static platter_t finger;
static array_t *first_unused;
static platter_t next_free_array_index;

inline static platter_t
get_reg (platter_t reg)
{
    assert(reg < NUM_REGS);
    return regs[reg];
}

inline static void
set_reg (platter_t reg, platter_t val)
{
    assert(reg < NUM_REGS);
    regs[reg] = val;
}

static void
init_arrays (void)
{
}

inline static int
array_used (platter_t array)
{
    if (array >= MAX_ARRAYS)
	return 0;

    array_t **a = (array_t**)arrays[array];

    if (a == 0)
	return 0;

    return a < arrays || a >= arrays + MAX_ARRAYS;
}

static platter_t
new_array (platter_t len)
{
    platter_t i;

    if (first_unused != 0)
    {
	i = (array_t**)first_unused - arrays;
	first_unused = *(array_t**)first_unused;
    }
    else
    {
	assert(next_free_array_index < MAX_ARRAYS);
	i = next_free_array_index++;
    }

    arrays[i] = (array_t*)malloc(sizeof(array_t) + sizeof(platter_t) * len);
    assert(arrays[i] != 0);

    arrays[i]->len = len;

    memset(arrays[i]->platters, 0, sizeof(platter_t) * len);

    return i;
}

static platter_t
array_len (platter_t array)
{
    assert(array_used(array));

    return arrays[array]->len;
}

static void
copy_array (platter_t dst, platter_t src)
{
    assert(array_used(dst) && array_used(src));
    assert(arrays[dst]->len == arrays[src]->len);

    memcpy(arrays[dst]->platters, arrays[src]->platters, sizeof(platter_t) * arrays[dst]->len);
}

inline static platter_t
array_index (platter_t array, platter_t index)
{
    assert(array_used(array));
    assert(index < arrays[array]->len);

    return arrays[array]->platters[index];
}

inline static void
array_amend (platter_t array, platter_t index, platter_t value)
{
    assert(array_used(array));
    assert(index < arrays[array]->len);

    arrays[array]->platters[index] = value;
}

static void
array_abandon (platter_t array)
{
    assert(array_used(array));

    free(arrays[array]);

    arrays[array] = (array_t*)first_unused;
    first_unused = (array_t*)&arrays[array];
}

static void
execute_insn (void)
{
    platter_t insn = array_index(0, finger);

    ++finger;

    switch (OPCODE(insn))
    {
	case 0 :		/* cmov */
	    if (get_reg(STD_C(insn)) != 0)
		set_reg(STD_A(insn), get_reg(STD_B(insn)));
	    break;

	case 1 :		/* ld */
	    set_reg(STD_A(insn), array_index(get_reg(STD_B(insn)), get_reg(STD_C(insn))));
	    break;

	case 2 :		/* st */
	    array_amend(get_reg(STD_A(insn)), get_reg(STD_B(insn)), get_reg(STD_C(insn)));
	    break;

	case 3 :		/* add */
	    set_reg(STD_A(insn), get_reg(STD_B(insn)) + get_reg(STD_C(insn)));
	    break;

	case 4 :		/* mul */
	    set_reg(STD_A(insn), get_reg(STD_B(insn)) * get_reg(STD_C(insn)));
	    break;

	case 5 :		/* div */
	    assert(get_reg(STD_C(insn)) != 0);

	    set_reg(STD_A(insn), get_reg(STD_B(insn)) / get_reg(STD_C(insn)));
	    break;

	case 6 :		/* nand */
	    set_reg(STD_A(insn), ~(get_reg(STD_B(insn)) & get_reg(STD_C(insn))));
	    break;

	case 7 :		/* halt */
	    exit(0);
	    break;

	case 8 :		/* alloc */
	    set_reg(STD_B(insn), new_array(get_reg(STD_C(insn))));
	    break;

	case 9 :		/* free */
	    array_abandon(get_reg(STD_C(insn)));
	    break;

	case 10 :		/* out */
	    output(get_reg(STD_C(insn)));
	    break;

	case 11 :		/* in */
	    {
		int val = input();

		if (val == -1)
		    set_reg(STD_C(insn), 0xffffffff);
		else
		{
		    assert(val >= 0 && val < 256);
		    set_reg(STD_C(insn), val);
		}
	    }
	    break;

	case 12 :		/* jmp */
	    {
		platter_t array = get_reg(STD_B(insn));
		if (array != 0)
		{
		    platter_t arr;

		    array_abandon(0);
		    arr = new_array(array_len(array));
		    assert(arr == 0);
		    copy_array(0, array);
		}

		finger = get_reg(STD_C(insn));
	    }
	    break;

	case 13 :		/* li */
	    set_reg(SPECIAL_A(insn), SPECIAL_VAL(insn));
	    break;

	default :
	    assert(0);
    }
}

void dump_arrays()
{
	FILE *arraydump=fopen("DUMPFILE","wb");
	int i;
	for (i = 0; i < MAX_ARRAYS; ++i)
                    {
                        if (array_used(i))
                            fwrite(arrays[i]->platters,array_len(i)*sizeof(platter_t),1,arraydump);
                    }
	fclose(arraydump);
}


static void
do_debug_mon (void)
{
    printf("\n");

    for (;;)
    {
	int c = input();

	switch (c)
	{
	    case 'a' :
		{
		    int i;

		    for (i = 0; i < MAX_ARRAYS; ++i)
		    {
			if (array_used(i))
			    printf("array %6d    len %9d\n", i, array_len(i));
		    }
		}
		break;

	    case 'n' :
		execute_insn();
		printf("\n");
		break;

	    case 'c' :
		return;

	    case 'f':
		flush_streams();
		break;

	    case 'd' :
		printf("dumping arrays.. ");
		dump_arrays();
		printf("done\n");
		break;

           case 'q' :
                exit(0);
                break;

	    default :
		printf("illegal command\n");
		break;
	}
    }
}

static void
run (void)
{
    for (;;)
    {
	if (debug_mon)
	{
	    do_debug_mon();
	    debug_mon = 0;
	}

	execute_insn();
    }
}

static void
break_handler (int sig)
{
    debug_mon = 1;
}

int
main (int argc, char *argv[])
{
    struct stat buf;
    FILE *file;
    platter_t len, i;
    int result;
    platter_t arr;

    if (argc != 4)
    {
	fprintf(stderr, "Usage: um <machine-file> <output-log> <input-log>\n");
	return 1;
    }

    result = stat(argv[1], &buf);
    assert(result == 0);

    assert((buf.st_size & 3) == 0);

    len = buf.st_size >> 2;

    init_arrays();

    arr = new_array(len);
    assert(arr == 0);

    file = fopen(argv[1], "rb");
    assert(file != 0);

    for (i = 0; i < len; ++i)
    {
	unsigned char bytes[4];

	result = fread(bytes, 4, 1, file);
	assert(result == 1);

	array_amend(0, i,
		    ((platter_t)bytes[0] << 24)
		    | ((platter_t)bytes[1] << 16)
		    | ((platter_t)bytes[2] << 8)
		    | ((platter_t)bytes[3]));
    }

    signal(SIGINT, break_handler);

    setup_files(0, 0, argv[2], argv[3]);

    run();

    return 0;
}
